#############################################################################
#############################################################################
#
# latexMenu.tcl (called from latex.tcl)
#
# LaTeX menu definitions and key bindings
#
#############################################################################
#
# Author:  Tom Scavo <trscavo@syr.edu>, Vince Darley <vince@santafe.edu>
#
#############################################################################
#############################################################################

proc latexMenu.tcl {} {}

namespace eval TeX::sub {}
namespace eval TeX::mp {}

# The highest level menu macro, to be called initially or whenever 
# the LaTeX menu must be rebuilt (as in 'shadowtexSig' in latex.tcl,
# or 'toggleLaTeXMenus' below): (simplified VMD July 97)
proc buildLaTeXMenuQuietly {} {
  global texMenu useShortLaTeXMenu useAMSLaTeX TeX::Menus
  toggleAMSLaTeXmenus
  TeX::sub::MathModes
  
  menu::buildProc texMenu buildLaTeXMenu
  menu::buildSome texMenu
}

# Toggle the short menu flag and rebuild the LaTeX menu.  If the
# latter operation fails, toggle the flag back to its original position.
proc toggleLaTeXMenus {} {
  toggleLaTeXMenuItem useShortLaTeXMenu
  buildLaTeXMenuQuietly
}
# Toggle the AMS-LaTeX flag, reload the macros and menu, and rebuild 
# the LaTeX menu.  If either of the latter two operations fail, toggle
# the flag back to its original position.
proc toggleAMSLaTeX {} {
  global useAMSLaTeX
  toggleLaTeXMenuItem useAMSLaTeX
  toggleAMSLaTeXmenus
  markMenuItem {LaTeX Utilities} {AMS-LaTeX} $useAMSLaTeX
  menu::buildSome "Math Style" "Text Size" "Math Environments"
}

proc toggleLaTeXMenuItem {flag} {
  global $flag
  set $flag [expr {1 - [set $flag]}]
  prefs::modified $flag
}

proc listToDummySubmenus {l} {
  foreach i $l {
    lappend ret [list Menu -n $i {}]
  }
  return $ret
}

# Return a short menu or a long menu, depending on the value of
# the global variable 'useShortLaTeXMenu':
proc buildLaTeXMenu {} {
  global useShortLaTeXMenu TeX::Menus texMenu
  if {$useShortLaTeXMenu} {
    return [list build [concat \
     [listToDummySubmenus [set TeX::Menus(Top)]] \
     [list "LaTeX Help"] \
     {(-} \
     [listToDummySubmenus [set TeX::Menus(Bottom)]] \
     [listToDummySubmenus [list "Text Submenus" "Math Submenus"]] \
     ] \
     {TeX::mp::latex -M TeX -m} \
     [concat [set TeX::Menus(Top)] [set TeX::Menus(Bottom)] \
     [list "Text Submenus" "Math Submenus"]] \
     $texMenu]
  } else {
    return [list build [concat \
     [listToDummySubmenus [set TeX::Menus(Top)]] \
     [list "LaTeX Help"] \
     {(-} \
     [listToDummySubmenus [set TeX::Menus(Bottom)]] \
     {(-} \
     [listToDummySubmenus [set TeX::Menus(TextSubMenus)]] \
     {(-} \
     [listToDummySubmenus [set TeX::Menus(MathSubMenus)]] \
     ] \
     {TeX::mp::latex -M TeX -m} \
     [concat [set TeX::Menus(Top)] [set TeX::Menus(Bottom)] \
     [set TeX::Menus(TextSubMenus)] [set TeX::Menus(MathSubMenus)]] \
     $texMenu]
  }
} 
menu::buildProc "Math Submenus" TeX::sub::Math
menu::buildProc "Text Submenus" TeX::sub::Text
menu::buildProc Process TeX::sub::Process
menu::buildProc ProcessOpen TeX::sub::ProcessOpen
proc TeX::sub::Math {} {
  global TeX::Menus
  return [list build \
   [listToDummySubmenus [set TeX::Menus(MathSubMenus)]] \
   {dummy -M TeX -m} \
   [set TeX::Menus(MathSubMenus)] \
   {Math Submenus}]
}
proc TeX::sub::Text {} {
  global TeX::Menus
  return [list build \
   [listToDummySubmenus [set TeX::Menus(TextSubMenus)]] \
   {dummy -M TeX -m} \
   [set TeX::Menus(TextSubMenus)] \
   {Text Submenus}]
}

ensureset TeX::Menus(TextSubMenus) [list \
 "Text Style" "Text Size" "Text Commands" "International" \
 "Environments" "Boxes" "Miscellaneous"\
 ]

ensureset TeX::Menus(Top) [list "Process" "Goto" "LaTeX Utilities"]

ensureset TeX::Menus(Bottom) [list "Documents" "Page Layout" \
 "Sectioning" "Theorem"]

ensureset TeX::Menus(MathSubMenus) [list \
 "Math Modes" "Math Style" "Math Environments" "Formulas" "Greek"  \
 "Binary Operators" "Relations" "Arrows" "Dots" "Symbols" \
 "Functions" "Large Operators" "Delimiters" "Math Accents" \
 "Grouping" "Spacing" \
 ]

proc TeX::mp::latex {menu item} {
  switch -- $item {
    "LaTeX Help" {
      help::openFile "LaTeX Help"
    }
    default {
      eval $item
    }
  }
}

#############################################################################
#
# Submenu definitions
#
#############################################################################

# Return the "Process" submenu.  If the current document belongs
# to a TeX fileset, display the base filename throughout.
#
############ New stuff FBO 2001-09 ##############
# New: TeX program submenu in Process to select the TeX program
# to use. Removed pdflatex xxx.tex since tex files are now "typeset".
# To pdflatex a file, choose pdftex as the TeX program and latex as 
# the TeX format before typesetting.
# If useNewerCMacTeXInterface is set, add a "Makeindex styles" submenu
# to select the style files use when make the index and the glossary.
proc TeX::sub::Process {{currentWin ""}} {
  global mode TeXmodeVars
  global texSig texAppSignatures
  if {$currentWin == ""} {set currentWin [win::TopNonProcessWindow]}
  if {$currentWin == ""} {
    return [TeX::sub::MinimalProcess]
  } elseif { $mode == "TeX" } {
    set currentDoc [file tail $currentWin]
    # Process an untitled window:
    if { [set num [winUntitled]] } {
      if { $num > 1 } { set currentDoc "untitled$num" }
      # fall through
    }
  } else {
    return [TeX::sub::MinimalProcess]
  }
  set docBasename [file rootname $currentDoc]
  set projBasename $docBasename
  set currentProj [isWindowInFileset $currentWin "tex"]
  if {$currentProj !=	""} {
    set theWin [texFilesetBaseName $currentProj]
    set currentDoc [file tail $theWin]
    set docBasename [file rootname $currentDoc]
  } else {
    set theWin $currentWin
  }
  
  # Determine which menu items are dimmed:
  set theRoot [file root $theWin]
  foreach ext {DVI PS AUX IDX GLO PDF} {
    if {[file exists "$theRoot.[string tolower $ext]"] || [file exists "$theRoot.$ext"]} {
      set prefix$ext "" 
    } else {
      set prefix$ext "\(" 
    }
  }
  # Selection may change before we have a chance to rebuild
  # this menu.  Hence always assume there is a selection.
  # With a '-postcommand' we could get this correct.
  set prefixSelection ""
  
  set ma [list \
   {Menu -n "TeX Format" {}} \
   {Menu -n "TeX program" {}} \
  ]
  # If we use CMacTeX >= 4.0, add a submenu to select the style
  # files to use in makeindex and makeglossary.
  if {$TeXmodeVars(useNewerCMacTeXInterface)} {
    lappend ma [list \
     Menu -n "MakeIndex styles" {} \
    ]
  }
  lappend ma "(-"

  if {$TeXmodeVars(runTeXInBack)} {
    lappend ma [list /TBack Typeset $currentDoc] \
     [list /T<U<ITypeset $currentDoc]
  } else {
    lappend ma [list /TTypeset $currentDoc] \
     [list /T<U<IBack Typeset $currentDoc]
  }
  if {[info exists texAppSignatures(Textures)] && \
   $texSig == $texAppSignatures(Textures)} {
    lappend ma "(-"\
     [list /`<BSynchronize $currentDoc] \
     "(-"
  }
  lappend ma \
   [list <U<O/V${prefixDVI}View "$docBasename\.dvi"] \
   [list <U<O/P${prefixDVI}Print "$docBasename\.dvi"] \
   "(-" \
   {<U<O/TTypeset Clipboard} \
   [list <U<I<O/T${prefixSelection}Typeset Selection] \
   "(-" \
   [list <S${prefixPS}Open "$docBasename\.ps"] \
   [list <S${prefixDVI}dvips "$docBasename\.dvi"] \
   [list ${prefixDVI}dvipdf "$docBasename\.dvi"] \
   [list ${prefixPS}Print "$docBasename\.ps"] \
   [list ${prefixPS}Distill "$docBasename\.ps"] \
   "(-" \
   [list ${prefixPS}View "$docBasename\.ps"] \
   [list <I<O/P${prefixPDF}View "$docBasename\.pdf"] \
   "(-" \
   [list <SOpen "$docBasename\.bbl"] \
   [list <S/B<I<O${prefixAUX}bibtex "$docBasename\.aux"] \
   [list <E<SOpen "$docBasename\.ind"] \
   [list <S${prefixIDX}makeindex "$docBasename\.idx"] \
   [list <E<SOpen "$docBasename\.glo"] \
   [list <S${prefixGLO}makeglossary "$docBasename\.glo"] \
   "(-" \
   {Menu -n "Open" {}} \
   "(-" \
   {Remove Auxiliary Files} \
   {Remove Temporary Files}
  return [list build $ma {TeX::mp::Processmenu -M TeX -m} \
   {{TeX Format} {TeX program} {MakeIndex styles} ProcessOpen}]
}

proc synchronizeDoc {args} {
  global Texturesconnections
  
  set TeXjob ""
  set currentWin [win::Current]
  if { $currentWin == "" } {
    return
  }
  
  # Strip off trailing garbage (if any):
  regexp {(.*) <[0-9]+>} $currentWin dummy currentWin
  # Is the current window part of TeX fileset?
  
  if {[info exists Texturesconnections]} {
    foreach entry $Texturesconnections {
      if {[lindex $entry 0] == $currentWin} {
        set TeXjob [lindex $entry 1]
        break
      }
    }
    
    AEBuild -r '*TEX' BSRs FFoc long [getPos] Jobi $TeXjob
    #AE::build::throw '*TEX' BSRs FFoc long [getPos] Jobi $TeXjob
  }
}

proc TeX::sub::MinimalProcess {} {
  set ma [list \
   [list /TTypeset] \
   "(-" \
   {<U<O/TTypeset Clipboard} \
   {Typeset Selection} \
   ]
  return [list build $ma {TeX::mp::Processmenu -M TeX -m}]
}

# Return a submenu of LaTeX auxiliary files with basename $basename1,
# except for the .aux file whose primary basename is $basename2.
#
proc TeX::sub::ProcessOpen {} {
  set currentWin [win::Current]
  set currentDoc [file tail $currentWin]
  set docBasename [file rootname $currentDoc]
  set projBasename $docBasename
  set currentProj [isWindowInFileset $currentWin "tex"]
  if {$currentProj !=	""} {
    set currentDoc [file tail [texFilesetBaseName $currentProj]]
    set docBasename [file rootname $currentDoc]
  }
  set ma [list \
   [list "$docBasename\.log"] \
   [list "<S$docBasename\.aux"] \
   [list "<S$projBasename\.aux"] \
   [list "$docBasename\.toc"] \
   [list "$docBasename\.lof"] \
   [list "$docBasename\.lot"] \
   [list "$docBasename\.idx"] \
   [list "$docBasename\.ind"] \
   [list "$docBasename\.bbl"] \
   "(-" \
   [list "$docBasename\.blg"] \
   [list "$docBasename\.ilg"] \
   "(-" \
   {<U<O/OAny TeX File} \
   ]
  return [list build $ma {TeX::mp::Processmenu -M TeX -m} \
   "" {Open}]
  ]
}

# pdflatex xyz.tex is no longer a menu item since you choose
# the TeX program (pdftex) and the format (latex) from the
# correspondng submenus of the Process menu.
proc TeX::mp::Processmenu {submenu item} {
  switch -regexp -- $item {
    {Typeset Selection}  {set func "typesetSelection"}
    {Typeset Clipboard}  {set func "typesetClipboard"}
    "Back\ Typeset.*"	{set func "typeset 1"}
    "Typeset.*"  {set func "typeset"}
    "Synchronize.*" {set func "synchronizeDoc"}
    "View.*\.dvi$"  {set func {doTypesetCommand view DVI}}
    "Print.*\.dvi$"  {set func {doTypesetCommand print DVI}}
    "dvips.*\.dvi$"  {set func {doTypesetCommand dvips DVI}}
    "dvipdf.*\.dvi$"  {set func {doTypesetCommand dvipdf DVI}}
    "View.*\.ps$"  {set func {doTypesetCommand view PS}}
    "Distill.*\.ps$"  {set func {doTypesetCommand distill PS}}
    "View.*\.pdf$"  {set func {doTypesetCommand view PDF}}
    "Print.*\.ps$"  {set func {doTypesetCommand print PS}}
    "bibtex.*"  {set func {doTypesetCommand bibtex AUX}}
    "makeindex.*"  {set func {doTypesetCommand makeindex IDX}}
    "makeglossary.*"  {set func {doTypesetCommand makeindex GLO}}
    ".*\.ps$"  {set func {doTypesetCommand open PS}}
    ".*\.bbl$"  {set func {doTypesetCommand open BBL}}
    ".*\.ind$"  {set func {doTypesetCommand open IND}}
    ".*\.log$"  {set func {doTypesetCommand open LOG}}
    ".*\.aux$"  {set func {doTypesetCommand open AUX 1}}
    ".*\.aux $"  {set func {doTypesetCommand open AUX}}
    ".*\.toc$"  {set func {doTypesetCommand open TOC}}
    ".*\.lof$"  {set func {doTypesetCommand open LOF}}
    ".*\.lot$"  {set func {doTypesetCommand open LOT}}
    ".*\.idx$"  {set func {doTypesetCommand open IDX}}
    ".*\.ind$"  {set func {doTypesetCommand open IND}}
    ".*\.blg$"  {set func {doTypesetCommand open BLG}}
    ".*\.ilg$"  {set func {doTypesetCommand open ILG}}
    ".*\.glo$"  {set func {doTypesetCommand open GLO}}
    {Any TeX File}  {set func "openAnyTeXFile"}
    {Remove Auxiliary Files}  {set func "removeAuxiliaryFiles"}
    {Remove Temporary Files}  {set func "removeTemporaryFiles"}
    default {set func $item}
  }
  eval $func
}

# Return the "Goto" submenu:
set menu::proc(Goto) TeX::mp::goto
set menu::otherflags(Goto) {-M TeX -m}
set menu::items(Goto) {
  "<U/SLaTeX"
  "BibTeX"
  "MakeIndex"
  "(-"
  {Next Template Stop}
  {Prev Template Stop}
  "(-"
  {Next Command}
  {Prev Command}
  {Next Command Select}
  {Prev Command Select}
  {Next Command Select With Args}
  {Prev Command Select With Args}
  "(-"
  {Next Environment}
  {Prev Environment}
  {Next Environment Select}
  {Prev Environment Select}
  "(-"
  {Next Section}
  {Prev Section}
  {Next Section Select}
  {Prev Section Select}
  {Next Subsection}
  {Prev Subsection}
  {Next Subsection Select}
  {Prev Subsection Select}
}

proc TeX::mp::goto {submenu item} {
  switch -- $item {
    "LaTeX"  {set func "texApp TeX"}
    "BibTeX" - "MakeIndex" {set func "texApp $item"}
    {Next Template Stop}  {set func "ring::+"}
    {Prev Template Stop}  {set func "ring::-"}
    {Next Command}  {set func "nextCommand"}
    {Prev Command}  {set func "prevCommand"}
    {Next Command Select}  {set func "nextCommandSelect"}
    {Prev Command Select}  {set func "prevCommandSelect"}
    {Next Command Select With Args}  {set func "nextCommandSelectWithArgs"}
    {Prev Command Select With Args}  {set func "prevCommandSelectWithArgs"}
    {Next Environment}  {set func "nextEnvironment"}
    {Prev Environment}  {set func "prevEnvironment"}
    {Next Environment Select}  {set func "nextEnvironmentSelect"}
    {Prev Environment Select}  {set func "prevEnvironmentSelect"}
    {Next Section}  {set func "nextSection"}
    {Prev Section}  {set func "prevSection"}
    {Next Section Select}  {set func "nextSectionSelect"}
    {Prev Section Select}  {set func "prevSectionSelect"}
    {Next Subsection}  {set func "nextSubsection"}
    {Prev Subsection}  {set func "prevSubsection"}
    {Next Subsection Select}  {set func "nextSubsectionSelect"}
    {Prev Subsection Select}  {set func "prevSubsectionSelect"}
    default {set func $item}
  }
  eval $func
}

# Return the "LaTeX Utilities" submenu:

set "menu::proc(LaTeX Utilities)" TeX::mp::latexUtils
set "menu::posteval(LaTeX Utilities)" {
  markMenuItem -m {LaTeX Utilities} {Short LaTeX Menu} $useShortLaTeXMenu
  markMenuItem -m {LaTeX Utilities} {AMS-LaTeX} $useAMSLaTeX
}
set "menu::otherflags(LaTeX Utilities)" {-M TeX -m}
set "menu::items(LaTeX Utilities)" {
  {Delete Template Stops}
  {Delete Comments}
  "(-"
  {Convert Quotes}
  {Convert Dollar Signs}
  "(-"
  {Short LaTeX Menu}
  {AMS-LaTeX}
}
#		{<U<O/CChoose Command}
#		"(-"

proc TeX::mp::latexUtils {submenu item} {
  switch -- $item {
    {Choose Command}  {set func {chooseCommand [getLaTeXMenu]}}
    {Insert Reference}  {set func "insertReference"}
    {Delete Template Stops}  {set func "ring::clear"}
    {Delete Comments}  {set func "deleteComments"}
    {Convert Quotes}  {set func "convertQuotes"}
    {Convert Dollar Signs}  {set func "convertDollarSigns"}
    {Short LaTeX Menu}  {set func "toggleLaTeXMenus"}
    {AMS-LaTeX}  {set func "toggleAMSLaTeX"}
    default {set func "TeX::[join $item {}]"}
  }
  eval $func
}

# Return the "Documents" submenu:
set "menu::proc(Documents)" TeX::mp::documents
set "menu::otherflags(Documents)" {-M TeX -m}
set "menu::items(Documents)" [list \
 {<U<O/NNew Document} \
 {Menu -n "Insert Document" {}} \
 "(-" \
 "options" \
 "<B<I/Uusepackage" \
 {Menu -n "Packages" {}} \
 "(-" \
 "filecontents" \
 {filecontents All} \
 "(-" \
 {Rebuild Documents Submenu} \
 ]
set menu::which_subs(Documents) [list "Insert Document" "Packages"]
menu::buildProc Packages TeX::sub::Packages

set "menu::proc(Insert Document)" TeX::mp::documents
set "menu::otherflags(Insert Document)" {-M TeX -m}
set "menu::items(Insert Document)" {
  "article"
  "report"
  "book"
  "letter"
  "slides"
  "other"
}


proc TeX::mp::documents {submenu item} {
  switch -- $item {
    {New Document}  {set func "newLaTeXDocument"}
    {article}  {set func "articleDocumentclass"}
    {report}  {set func "reportDocumentclass"}
    {book}  {set func "bookDocumentclass"}
    {letter}  {set func "letterDocumentclass"}
    {slides}  {set func "slidesDocumentclass"}
    {other}  {set func "otherDocumentclass"}
    {usepackage} {insertPackage ""; return}
    {filecontents All} {set func "filecontentsAll"}
    {Rebuild Documents Submenu}  {
      global packagesSubmenuItems
      set packagesSubmenuItems ""
      message "Rebuilding the Documents submenu"
      menu::buildSome Documents
      message ""
      return
    }
    default {set func $item}
  }
  eval $func
}
proc TeX::sub::Packages {} {
  global TeXmodeVars
  if { $TeXmodeVars(buildPkgsSubmenu) } {
    global packagesSubmenuItems
    if { ![info exists packagesSubmenuItems] || $packagesSubmenuItems == "" } {
      global latexPackages
      set folders [buildTeXSearchPath 0]
      set packagesSubmenuItems \
       [menu::buildHierarchy $folders "Packages" TeX::mp::package latexPackages ".sty"]
    }
    return $packagesSubmenuItems
  } else {
    catch {unset packagesSubmenuItems}
    return ""
  }
}
proc TeX::mp::package {submenu pkgName} {
  global latexPackages
  # See latexMacros.tcl for definition of 'insertPackage':
  insertPackage [file tail [file rootname $latexPackages([file join $submenu $pkgName])]]
}

# Return the "Page Layout" submenu:

set "menu::otherflags(Page Layout)" {-M TeX -m}
set "menu::items(Page Layout)"  {
  "maketitle"
  "(-"
  "abstract"
  "titlepage"
  "(-"
  "pagestyle"
  "thispagestyle"
  "pagenumbering"
  "(-"
  "twocolumn"
  "onecolumn"
}


# Return the "Sectioning" submenu:

set menu::proc(Sectioning) TeX::mp::sectioning
set menu::otherflags(Sectioning) {-M TeX -m}

# need to integrate these two
set menu::items(Sectioning) {
  "<Spart"
  "<S<Upart*"
  "<Spart with label"
  "<E<Schapter"
  "<S<Uchapter*"
  "<Schapter with label"
  "<E<Ssection"
  "<S<Usection*"
  "<Ssection with label"
  "<E<Ssubsection"
  "<S<Usubsection*"
  "<Ssubsection with label"
  "<E<Ssubsubsection"
  "<S<Usubsubsection*"
  "<Ssubsubsection with label"
  "<E<Sparagraph"
  "<S<Uparagraph*"
  "<Sparagraph with label"
  "<E<Ssubparagraph"
  "<S<Usubparagraph*"
  "<Ssubparagraph with label"
  "(-"
  "appendix"
}

# proc TeX::mp::sectioning {submenu item} {
#   if {[regexp (part|chapter|section|paragraph) $item]} {
#     eval sectioning $item
#   } else {
#     eval $item
#   }
# }
proc TeX::mp::sectioning {submenu item} {
  if {$item == "appendix"}  {
    appendix
  } else {
    switch -- $item {
      {part with label}  {
        set secName "part"
        set secLabel "part[TeX::labelDelim]"
      }
      {chapter with label}  {
        set secName "chapter"
        set secLabel "chap[TeX::labelDelim]"
      }
      {section with label}  {
        set secName "section"
        set secLabel "sec[TeX::labelDelim]"
	    }
      {subsection with label}  {
        set secName "subsection"
        set secLabel "subsec[TeX::labelDelim]"
      }
      {subsubsection with label}  {
        set secName "subsubsection"
        set secLabel "subsubsec[TeX::labelDelim]"
      }
      {paragraph with label}  {
        set secName "paragraph"
        set secLabel "par[TeX::labelDelim]"
      }
      {subparagraph with label}  {
        set secName "subparagraph"
        set secLabel "subpar[TeX::labelDelim]"
      }
      default {
        set secName  $item
        set secLabel ""
      }
    }
    append leftWrap [openingCarriageReturn] \
     "\\${secName}\{"
    if {$secLabel == ""} {
      append rightWrap "\}\r"
    } else {
      append rightWrap "\}\\label\{" $secLabel "\}\r"
    }
    append rightWrap "" [closingCarriageReturn]
    if {$secLabel == ""} {
      if [elec::Wrap $leftWrap $rightWrap] {
        message "sectioning done"
      } else {
        message "enter the $secName"
      }
    } else {
      if [elec::Wrap $leftWrap $rightWrap] {
        message "enter the label"
      } else {
        message "enter the $secName, press <tab>, enter the label"
      }
    }
  }
}

# Return the "Text Style" submenu:
set "menu::otherflags(Text Style)" {-M TeX -m}
set "menu::items(Text Style)"  {
  "<S<B<I/Eemph"
  "<S<U<B<I/Eem"
  "<B<O/Uunderline"
  "(-"
  "<Supshape"
  "<Stextup"
  "<E<S<B<I/Itextit"
  "<S<U<B<I/Iitshape"
  "<E<S<B<I/Stextsl"
  "<S<U<B<I/Sslshape"
  "<E<S<B<I/Htextsc"
  "<S<U<B<I/Hscshape"
  "(-"
  "<Smdseries"
  "<Stextmd"
  "<E<S<B<I/Btextbf"
  "<S<U<B<I/Bbfseries"
  "(-"
  "<S<B<I/Rtextrm"
  "<S<U<B<I/Rrmfamily"
  "<E<S<B<I/Wtextsf"
  "<S<U<B<I/Wsffamily"
  "<E<S<B<I/Ytexttt"
  "<S<U<B<I/Yttfamily"
  "(-"
  "<Snormalfont"
  "<Stextnormal"
}


# Return the "Text Size" submenu:
proc TeX::mp::textSize {menu item} {
  doTextSize $item
}

proc TeX::mp::textCommands {menu item} {
  insertObject "\\${item}"
}
set "menu::proc(Text Commands)" TeX::mp::textCommands
set "menu::otherflags(Text Commands)" {-M TeX -m}
set "menu::items(Text Commands)" {
  "textsuperscript"
  "textcircled"
  "(-"
  "textcompwordmark"
  "textvisiblespace"
  "(-"
  "!textemdash"
  "!textendash"
  "!textexclamdown"
  "!textquestiondown"
  "!textquotedblleft"
  "!textquotedblright"
  "!textquoteleft"
  "!textquoteright"
  "(-"
  "textbullet"
  "textperiodcentered"
  "(-"
  "textbackslash"
  "textbar"
  "textless"
  "textgreater"
  "(-"
  "textasciicircum"
  "textasciitilde"
  "(-"
  "textregistered"
  "texttrademark"
}

# Return the "International" submenu:
set menu::proc(International) TeX::mp::internat
set menu::otherflags(International) {-M TeX -m}
set menu::items(International) {
  {}
  {}
  {}
  {}
  {}
  "(-"
  {}
  {}
  {}
  {}
  {}
  "(-"
  {}
  {}
  {}
  {}
  {}
  "(-"
  ss
  SS
  "(-"
  {}
  {}
}
proc TeX::mp::internat {menu item} {
  $item
}

# Return the "Environments" submenu:

set menu::proc(Environments) TeX::mp::envs
set menu::otherflags(Environments) {-M TeX -m}
set menu::items(Environments) {
  "<I/kitemize"
  "<U<I/kenumerate"
  "<B<I/kdescription"
  "thebibliography"
  "(-"
  "<I/lslide"
  "<U<I/loverlay"
  "<B<I/lnote"
  "(-"
  "<I/mfigure"
  "<U<I/mtable"
  "<B<I/mtabular"
  "(-"
  "<I/nverbatim"
  "<U<I/nquote"
  "<B<I/nquotation"
  "verse"
  "(-"
  "<I/ocenter"
  "<U<I/oflushleft"
  "<B<I/oflushright"
  "(-"
  {<U<O/EChoose Environment}
}
proc TeX::mp::envs {submenu item} {
  switch -- $item {
    {Choose Environment}  {set func chooseEnvironment}
    default {set func $item}
  }
  eval $func
}

# Return the "Boxes" submenu:

set "menu::otherflags(Boxes)" {-M TeX -m}
set "menu::items(Boxes)"  {
  "<B<I/Mmbox"
  "makebox"
  "fbox"
  "framebox"
  "(-"
  "newsavebox"
  "sbox"
  "savebox"
  "usebox"
  "(-"
  "raisebox"
  "(-"
  "parbox"
  "minipage"
  "(-"
  "rule"
}

# Return the "Miscellaneous" submenu:
set menu::proc(Miscellaneous) TeX::mp::misc
set menu::otherflags(Miscellaneous) {-M TeX -m}
set menu::items(Miscellaneous) {
  "<B<I/Vverb"
  "<B<I/Ffootnote"
  {<B<I/Nmarginal note}
  "(-"
  "<B<I/Llabel"
  "<B<I/Xref"
  "eqref"
  "<B<I/Ppageref"
  "<B<I/Ccite"
  "<U<B<I/Cnocite"
  "(-"
  "<B<I/Jitem"
  "(-"
  "<B<I/'quotes"
  {<U<B<I/'double quotes}
  "(-"
  {TeX logo}
  {LaTeX logo}
  {LaTeX2e logo}
  "date"
  "(-"
  "!dag"
  "ddag"
  {!section mark}
  {!paragraph mark}
  "!copyright"
  "!pounds"
}
proc TeX::mp::misc {submenu item} {
  switch -- $item {
    {marginal note}  {set func "marginalNote"}
    "item" {set func "insertItem"}
    "label" {set func "insertLabel"}
    {double quotes}  {set func "dblQuotes"}
    {TeX logo}  {set func "texLogo"}
    {LaTeX logo}  {set func "latexLogo"}
    {LaTeX2e logo}  {set func "latex2eLogo"}
    "date" {set func "today"}
    {section mark}  {set func "sectionMark"}
    {paragraph mark}  {set func "paragraphMark"}
    default {set func $item}
  }
  eval $func
}

# Return a "Math Modes" submenu (called below and in latexKeys.tcl):
set "menu::proc(Math Modes)" TeX::mp::mathModes
set "menu::otherflags(Math Modes)" {-M TeX -m}
proc TeX::sub::MathModes {} {
  global TeXmodeVars menu::items
  if {$TeXmodeVars(useDollarSigns)} {
    Bind '4' <zc> texMath "TeX"
    Bind '4' <zoc> texDisplaymath "TeX"
    set "menu::items(Math Modes)" {
      {<B<O/MTeX math}
      {<B<I<O/MTeX displaymath}
      "(-"
      {LaTeX math}
      {LaTeX displaymath}
    }
  } else {
    Bind '4' <zc> latexMath "TeX"
    Bind '4' <zoc> latexDisplaymath "TeX"
    set "menu::items(Math Modes)" {
      {TeX math}
      {TeX displaymath}
      "(-"
      {<B<O/MLaTeX math}
      {<B<I<O/MLaTeX displaymath}
    }
  }
}
proc TeX::mp::mathModes {submenu item} {
  switch -- $item {
    {TeX math}  {set func "texMath"}
    {TeX displaymath}  {set func "texDisplaymath"}
    {LaTeX math}  {set func "latexMath"}
    {LaTeX displaymath}  {set func "latexDisplaymath"}
    default {set func $item}
  }
  eval $func
}

proc TeX::mp::mathStyle {submenu item} {
  switch -- $item {
    mathbb {doUppercaseMathStyle mathbb "math blackboard bold"}
    mathfrak {doMathStyle mathfrak "math fraktur"}
    mathit {doMathStyle mathit "math italic"}
    mathrm {doMathStyle mathrm "math roman"}
    mathbf {doMathStyle mathbf "math bold"}
    mathsf {doMathStyle mathsf "math sans serif"}
    mathtt {doMathStyle mathtt "math typewriter"}
    mathcal {doUppercaseMathStyle mathcal "math calligraphic"}
    displaystyle {doMathStyle displaystyle "display style"}
    textstyle {doMathStyle textstyle "text style"}
    scriptstyle {doMathStyle scriptstyle "script style"}
    scriptscriptstyle {doMathStyle scriptscriptstyle "scriptscript style"}
  }
}

# Return the "Math Style" submenu:
set "menu::proc(Math Style)" TeX::mp::mathStyle
set "menu::otherflags(Math Style)" {-M TeX -m}
set "menu::proc(Text Size)" TeX::mp::textSize
set "menu::otherflags(Text Size)" {-M TeX -m}
proc toggleAMSLaTeXmenus {} {
  global useAMSLaTeX menu::items
  if { $useAMSLaTeX } {
    set "menu::items(Math Style)" {
      "<B<I<O/Imathit"
      "<B<I<O/Rmathrm"
      "<B<I<O/Bmathbf"
      "<B<I<O/Wmathsf"
      "<B<I<O/Ymathtt"
      "<B<I<O/Cmathcal"
      "(-"
      "<B<I<O/Zmathbb"
      "mathfrak"
      "(-"
      "<B<I<O/Ddisplaystyle"
      "<B<I<O/Ttextstyle"
      "<B<I<O/Sscriptstyle"
      "scriptscriptstyle"
    }
    set "menu::items(Text Size)" {
      "<B<I/1Tiny"
      "<B<I/2tiny"
      "<B<I/3SMALL"
      "<B<I/4Small"
      "<B<I/5small"
      "normalsize"
      "<B<I/6large"
      "<B<I/7Large"
      "<B<I/8LARGE"
      "<B<I/9huge"
      "<B<I/0Huge"
    }
    set "menu::items(Math Environments)" {
      "<I/imath"
      "(-"
      "<S<B/iequation*"
      "<S<I<B/iequation"
      "subequations"
      "(-"
      "<S<B/jalign*"
      "<S<B<I/jalign"
      "<E<Sflalign"
      "<Sflalign*"
      "<E<Salignat"
      "<Salignat*"
      "<E<S<U/jgather*"
      "<S<I<U/jgather"
      "<E<S<B<U/jmultline*"
      "<S<I<B<U/jmultline"
      "(-"
      "gathered"
      "aligned"
      "alignedat"
      "split"
      "cases"
      "(-"
      "array"
      "subarray"
      "matrix"
      "pmatrix"
      "bmatrix"
      "Bmatrix"
      "vmatrix"
      "Vmatrix"
      "smallmatrix"
      "(-"
      {<U<O/EChoose Environment}
    }
  } else {
    set "menu::items(Math Style)" {
      "<B<I<O/Imathit"
      "<B<I<O/Rmathrm"
      "<B<I<O/Bmathbf"
      "<B<I<O/Wmathsf"
      "<B<I<O/Ymathtt"
      "<B<I<O/Cmathcal"
      "(-"
      "<B<I<O/Ddisplaystyle"
      "<B<I<O/Ttextstyle"
      "<B<I<O/Sscriptstyle"
      "scriptscriptstyle"
    }
    set "menu::items(Text Size)" {
      "<B<I/1tiny"
      "<B<I/2scriptsize"
      "<B<I/3footnotesize"
      "<B<I/4small"
      "<B<I/5normalsize"
      "<B<I/6large"
      "<B<I/7Large"
      "<B<I/8LARGE"
      "<B<I/9huge"
      "<B<I/0Huge"
    }
    set "menu::items(Math Environments)" {
      "<I/imath"
      "(-"
      "<U<I/idisplaymath"
      "<B<I/iequation"
      "(-"
      "<B<I/jeqnarray"
      "<U<I/jeqnarray*"
      "(-"
      "<I/jarray"
      "(-"
      {<U<O/EChoose Environment}
    }
  }
}

# Return the "Math Environments" submenu:
set "menu::proc(Math Environments)" TeX::mp::mathEnvs
set "menu::otherflags(Math Environments)" {-M TeX -m}
proc TeX::mp::mathEnvs {submenu item} {
  switch -- $item {
    "multline" - "multline\*" - "gather" - "gather\*" - \
     "align" -  "align\*" - "flalign" - "flalign\*" - "alignat" - "alignat\*" - \
     "array" - "subarray" - \
     "eqnarray\*" - "eqnarray" - \
     "split" - "aligned" - "gathered" - "alignedat" - "cases" - \
     "matrix" - "pmatrix" - "bmatrix" - "Bmatrix" - "vmatrix" - "Vmatrix" - \
     "smallmatrix" {
      set func "TeXmathenv $item"
    }
    "equation" {
      set func "mathEnvironment $item"
    }
    {Choose Environment}  {set func chooseEnvironment}
    default {set func $item}
  }
  eval $func
}

#--------------------------------------------------------------------------
# Adding a "Theorem" submenu: many thanks to...
#
# Paul Gastin
# LIAFA
# Universite Paris 7
# 2, place Jussieu
# F-75251 Paris Cedex 05
# email : Paul.Gastin@liafa.jussieu.fr
# www : http://www.liafa.jussieu.fr/~gastin
#--------------------------------------------------------------------------

# Return the "Theorem" submenu:
set menu::proc(Theorem) TeX::mp::theorem
set menu::otherflags(Theorem) {-M TeX -m}
set menu::items(Theorem) {
  "<E<Sdefinition"
  "<Sdefinition with label"
  "<E<Sremark"
  "<Sremark with label"
  "(-"
  "<E<Slemma"
  "<Slemma with label"
  "<E<Sproposition"
  "<Sproposition with label"
  "<E<Stheorem"
  "<Stheorem with label"
  "<E<Scorollary"
  "<Scorollary with label"
  "(-"
  "claim"
  "<E<Sclaimno"
  "<Sclaimno with label"
  "(-"
  "proof"
  "proofof"
}

proc TeX::mp::theorem {submenu item} {
  if {$item == "proofof"} {
    if {[wrapStructure "\\begin\{proofof\}\{\}" "" \
     "\\end\{proofof\}\r"] } {
      set msgText "selection wrapped, enter ref of proofof environment"
    } else {
      set msgText "enter ref of proofof environment, press <tab>, enter the body of proofof environment"
    }
  } else {
    switch -- $item {
	    {definition with label}  {
        set envName "definition"
        set envLabel "def[TeX::labelDelim]"
      }
      {remark with label}  {
        set envName "remark"
        set envLabel "rem[TeX::labelDelim]"
      }
      {lemma with label}  {
        set envName "lemma"
        set envLabel "lem[TeX::labelDelim]"
      }
      {proposition with label}  {
        set envName "proposition"
        set envLabel "prop[TeX::labelDelim]"
      }
      {theorem with label}  {
        set envName "theorem"
        set envLabel "thm[TeX::labelDelim]"
      }
      {corollary with label}  {
        set envName "corollary"
        set envLabel "cor[TeX::labelDelim]"
      }
      {claimno with label}  {
        set envName "claimno"
        set envLabel "claim[TeX::labelDelim]"
      }
      default {
        set envName  $item
        set envLabel ""
      }
    }
    if {$envLabel == ""} {
      append begStruct "\\begin\{" $envName "\}"
      append endStruct "\\end\{" $envName "\}\r"
      if {[wrapStructure $begStruct "" $endStruct]} {
        set msgText "selection wrapped"
      } else {
        set msgText "enter the body of $envName environment"
      }
    } else {
      append begStruct "\\begin\{" $envName "\}\\label\{" $envLabel "\}"
      append endStruct "\\end\{" $envName "\}\r"
      if {[wrapStructure $begStruct "" $endStruct]} {
        set msgText "selection wrapped, enter the label"
      } else {
        set msgText "enter the label, press <tab>, enter the body of $envName environment"
      }
    }
  }
  message $msgText
}

# Return the "Formulas" submenu:
set menu::proc(Formulas) TeX::mp::formulas
set menu::otherflags(Formulas) {-M TeX -m}
set menu::items(Formulas) {
  "subscript"
  "superscript"
  "(-"
  "<B<O/Ffrac"
  {<B<O/Rsqrt}
  {nth root}
  "(-"
  {<B<O/1one parameter}
  {<B<O/2two parameters}
}
proc TeX::mp::formulas {submenu item} {
  switch -- $item {
    {frac}  {set func "fraction"}
    {sqrt}  {set func "squareRoot"}
    {nth root}  {set func "nthRoot"}
    {one parameter}  {set func "oneParameter"}
    {two parameters}  {set func "twoParameters"}
    default {set func $item}
  }
  eval $func
}

proc TeX::mp::greek {submenu item} {
  checkMathMode $item 1; insertObject "\\$item"
}

# Return the "Greek" submenu:
set menu::proc(Greek) TeX::mp::greek
set menu::otherflags(Greek) {-M TeX -m}
set menu::items(Greek) {
  "alpha"
  "beta"
  "<SGamma"
  "<Sgamma"
  "<E<SDelta"
  "<Sdelta"
  "epsilon"
  "zeta"
  "eta"
  "<STheta"
  "<Stheta"
  "iota"
  "kappa"
  "<SLambda"
  "<Slambda"
  "mu"
  "nu"
  "<SXi"
  "<Sxi"
  "omicron"
  "<SPi"
  "<Spi"
  "rho"
  "<SSigma"
  "<Ssigma"
  "tau"
  "<SUpsilon"
  "<Supsilon"
  "<E<SPhi"
  "<Sphi"
  "chi"
  "<SPsi"
  "<Spsi"
  "<E<SOmega"
  "<Somega"
  "(-"
  "varepsilon"
  "vartheta"
  "varpi"
  "varrho"
  "varsigma"
  "varphi"
}

proc TeX::mp::binaryOperators {submenu item} {
  regsub "!." $item "" item
  if {[regexp {[lr]hd$} $item]} {
    if {![TeX::RequirePackage latexsym]} {return}
  }
  checkMathMode $item 1; insertObject "\\$item"
}

# Return the "Binary Operators" submenu:
set "menu::proc(Binary Operators)" TeX::mp::binaryOperators
set "menu::otherflags(Binary Operators)" {-M TeX -m}
set "menu::items(Binary Operators)" {
  "/+<I<U!pm"
  "mp"
  "times"
  "!div"
  "ast"
  "star"
  "circ"
  "bullet"
  "cdot"
  "cap"
  "cup"
  "uplus"
  "sqcap"
  "sqcup"
  "vee"
  "wedge"
  "setminus"
  "(-"
  "diamond"
  "bigtriangleup"
  "bigtriangledown"
  "triangleleft"
  "triangleright"
  "lhd"
  "rhd"
  "unlhd"
  "unrhd"
  "(-"
  "oplus"
  "ominus"
  "otimes"
  "oslash"
  "odot"
  "(-"
  "bigcirc"
  "dagger"
  "ddagger"
  "amalg"
  "wr"
}

# Return the "Relations" submenu:
set menu::proc(Relations) TeX::mp::relations
set menu::otherflags(Relations) {-M TeX -m}
set menu::items(Relations) {
  "<S!geq"
  "<S!leq"
  "<E<Ssucc"
  "<Sprec"
  "<E<Ssucceq"
  "<Spreceq"
  "<E<S!gg"
  "<S!ll"
  "<E<Ssupset"
  "<Ssubset"
  "<E<Ssupseteq"
  "<Ssubseteq"
  "<E<Ssqsupset"
  "<Ssqsubset"
  "<E<Ssqsupseteq"
  "<Ssqsubseteq"
  "<E<Sni"
  "<Sin"
  "<E<Sdashv"
  "<Svdash"
  "(-"
  "equiv"
  "sim"
  "simeq"
  "asymp"
  "!approx"
  "cong"
  "/=<I!neq"
  "doteq"
  "propto"
  "(-"
  "models"
  "perp"
  "mid"
  "parallel"
  "bowtie"
  "join"
  "smile"
  "frown"
}
proc TeX::mp::relations {submenu item} {
  # workaround alpha menu bug
  regsub "!." $item "" item
  if {[lsearch -exact "join sqsubset sqsupset" $item] != -1} {
    if {![TeX::RequirePackage latexsym]} {
      return
    }
  }
  checkMathMode $item 1; insertObject "\\$item"
}

# Return the "Arrows" submenu:
set menu::otherflags(Arrows) {-M TeX -m}
set menu::proc(Arrows) TeX::mp::arrows
set menu::items(Arrows) {
  "<SLeftarrow"
  "<Sleftarrow"
  "<E<SRightarrow"
  "<Srightarrow"
  "<E<SLeftrightarrow"
  "<Sleftrightarrow"
  "<E<SLongleftarrow"
  "<Slongleftarrow"
  "<E<SLongrightarrow"
  "<Slongrightarrow"
  "<E<SLongleftrightarrow"
  "<Slongleftrightarrow"
  "(-"
  "<SUparrow"
  "<Suparrow"
  "<E<SDownarrow"
  "<Sdownarrow"
  "<E<SUpdownarrow"
  "<Supdownarrow"
  "(-"
  "mapsto"
  "longmapsto"
  "leadsto"
  "(-"
  "leftharpoonup"
  "rightharpoonup"
  "leftharpoondown"
  "rightharpoondown"
  "rightleftharpoons"
  "hookleftarrow"
  "hookrightarrow"
  "nearrow"
  "searrow"
  "swarrow"
  "nwarrow"
}

proc TeX::mp::arrows {submenu item} {
  checkMathMode $item 1; insertObject "\\$item"
}

# Return the "Dots" submenu:
set menu::proc(Dots) TeX::mp::generalMath
set menu::otherflags(Dots) {-M TeX -m}
set menu::items(Dots) {
  "bullet"
  "cdot"
  "(-"
  "ldots"
  "cdots"
  "vdots"
  "ddots"
}

# Return the "Symbols" submenu:
set menu::proc(Symbols) TeX::mp::generalMath
set menu::otherflags(Symbols) {-M TeX -m}
set menu::items(Symbols) {
  "aleph"
  "hbar"
  "imath"
  "jmath"
  "ell"
  "wp"
  "Re"
  "Im"
  "mho"
  "(-"
  "angle"
  "backslash"
  "bot"
  "/0<Iemptyset"
  "exists"
  "forall"
  "!infty"
  "nabla"
  "!neg"
  "!partial"
  "prime"
  "!surd"
  "top"
  "(-"
  "Box"
  "Diamond"
  "triangle"
  "clubsuit"
  "diamondsuit"
  "heartsuit"
  "spadesuit"
  "(-"
  "flat"
  "natural"
  "sharp"
}

proc TeX::mp::generalMath {submenu item} {
  if {[lsearch -exact "mho Box Diamond" $item] != -1} {
    if {![TeX::RequirePackage latexsym]} {return}
  }
  switch -- $item {
    "lim" - "inf" - "liminf" - "limsup" - "max" - "min" - "sup" {
      checkMathMode $item 1
      if {[elec::Wrap "\\${item}_{" "}"]} {
        message "limit set"
      } else {
        message "enter limit"
	    }
    }
    "pmod" {
      checkMathMode "pmod" 1
      if {[elec::Wrap "\\pmod{" "}"]} {
        message "parenthesized mod set"
      } else {
        message "enter formula"
      }
    }
    "ldots" {insertObject "\\ldots"}
    default {
      checkMathMode $item 1; insertObject "\\$item"
    }
  }
}

# Return the "Functions" submenu:
set menu::proc(Functions) TeX::mp::generalMath
set menu::otherflags(Functions) {-M TeX -m}
set menu::items(Functions) {
  "arccos"
  "arcsin"
  "arctan"
  "arg"
  "cos"
  "cosh"
  "cot"
  "coth"
  "csc"
  "deg"
  "det"
  "dim"
  "exp"
  "gcd"
  "hom"
  "inf"
  "ker"
  "lg"
  "<B<O/Llim"
  "liminf"
  "limsup"
  "ln"
  "log"
  "max"
  "min"
  "Pr"
  "sec"
  "sin"
  "sinh"
  "sup"
  "tan"
  "tanh"
  "(-"
  "bmod"
  "pmod"
}

# Return the "Large Operators" submenu:
set "menu::otherflags(Large Operators)" {-M TeX -m}
set "menu::items(Large Operators)"  {
  "<B<O/Ssum"
  "<B<O/Pprod"
  "coprod"
  "<B<O/Iint"
  "oint"
  "(-"
  "bigcup"
  "bigcap"
  "bigsqcup"
  "bigvee"
  "bigwedge"
  "bigodot"
  "bigotimes"
  "bigoplus"
  "biguplus"
}

# Return the "Delimiters" submenu:
set menu::proc(Delimiters) TeX::mp::delimiters
set menu::otherflags(Delimiters) {-M TeX -m}
set menu::items(Delimiters) {
  "parentheses"
  "brackets"
  "braces"
  {vertical bars}
  {other delims}
  "(-"
  {half-open interval}
  {half-closed interval}
  "(-"
  {<Smulti-line big parentheses}
  {<Sbig parentheses}
  {<E<Smulti-line big brackets}
  {<Sbig brackets}
  {<E<Smulti-line big braces}
  {<Sbig braces}
  {<E<Smulti-line big vertical bars}
  {<Sbig vertical bars}
  {<E<Sother multi-line big delims}
  {<Sother big delims}
  "(-"
  {<Smulti-line big left brace}
  {<Sbig left brace}
  {<E<Sother multi-line mixed big delims}
  {<Sother mixed big delims}
}
proc TeX::mp::delimiters {submenu item} {
  switch -- $item {
    {vertical bars}  {set func "absoluteValue"}
    {other delims}  {set func "otherDelims"}
    {half-open interval}  {set func "half-openInterval"}
    {half-closed interval}  {set func "half-closedInterval"}
    {big parentheses}  {set func "bigParens"}
    {multi-line big parentheses}  {set func "multiBigParens"}
    {big brackets}  {set func "bigBrackets"}
    {multi-line big brackets}  {set func "multiBigBrackets"}
    {big braces}  {set func "bigBraces"}
    {multi-line big braces}  {set func "multiBigBraces"}
    {big vertical bars}  {set func "bigAbsValue"}
    {multi-line big vertical bars}  {set func "multiBigAbsValue"}
    {other big delims}  {set func "otherBigDelims"}
    {other multi-line big delims}  {set func "otherMultiBigDelims"}
    {big left brace}  {set func "bigLeftBrace"}
    {multi-line big left brace}  {set func "multiBigLeftBrace"}
    {other mixed big delims}  {set func "otherMixedBigDelims"}
    {other multi-line mixed big delims}  {set func "otherMultiMixedBigDelims"}
    default {set func $item}
  }
  eval $func
}

# Return the "Math Accents" submenu:
set "menu::otherflags(Math Accents)" {-M TeX -m}
set "menu::items(Math Accents)"  {
  "<B<O/Aacute"
  "<B<O/Bbar"
  "breve"
  "<B<O/Ccheck"
  "<B<O/Ddot"
  "ddot"
  "<B<O/Ggrave"
  "<B<O/Hhat"
  "<B<O/Ttilde"
  "<B<O/Vvec"
  "(-"
  "widehat"
  "widetilde"
}

# Return the "Grouping" submenu:
set "menu::otherflags(Grouping)" {-M TeX -m}
set "menu::items(Grouping)"  {
  "<B<O/Uunderline"
  "<B<O/Ooverline"
  "<B<I<O/Uunderbrace"
  "<B<I<O/Ooverbrace"
  "(-"
  "overrightarrow"
  "overleftarrow"
  "(-"
  "stackrel"
}

# Return the "Spacing" submenu:
set menu::proc(Spacing) TeX::mp::spacing
set menu::otherflags(Spacing) {-M TeX -m}
set menu::items(Spacing) {
  {neg thin}
  "thin"
  "medium"
  "thick"
  "(-"
  "quad"
  "qquad"
  "(-"
  "hspace"
  "vspace"
  "(-"
  "hfill"
  "vfill"
  "(-"
  "smallskip"
  "medskip"
  "bigskip"
}
proc TeX::mp::spacing {submenu item} {
  switch -- $item {
    {neg thin}  {set func "negThin"}
    default {set func $item}
  }
  eval $func
}

############ New stuff FBO 2001-09 ##############
#
set "menu::proc(TeX program)" TeX::ChooseProgram
set "menu::otherflags(TeX program)" {-M TeX -m}
set "menu::posteval(TeX program)" {
  TeX::SetProgram $TeXmodeVars(nameOfTeXProgram)
}

set "menu::proc(TeX Format)" TeX::Format
set "menu::otherflags(TeX Format)" {-M TeX -m}
set "menu::posteval(TeX Format)" {
  global TeXmodeVars
  
  if {$TeXmodeVars(noFormatNameInFile)} {
    TeX::SetFormat $TeXmodeVars(nameOfTeXFormat)
  }
}

proc TeX::sub::MakeIndex {} {
  global TeX::indexStyle TeX::glossaryStyle
  
  return [list build \
   [list \
    "Index style " \
    "No index style" \
    "${TeX::indexStyle}" \
    "(-" \
    "Glossary style " \
    "No glossary style" \
    "${TeX::glossaryStyle}" \
   ] \
   {TeX::ChooseStyle -M TeX -m} \
   {} \
   {} \
  ]
}
menu::buildProc {MakeIndex styles} TeX::sub::MakeIndex

set "menu::posteval(MakeIndex styles)" {
  global TeX::indexStyle TeX::glossaryStyle
  
  enableMenuItem {MakeIndex styles} "${TeX::indexStyle}" off
  enableMenuItem {MakeIndex styles} "${TeX::glossaryStyle}" off
}

# These procs update the menus according to the mode prefs.
proc TeX::updateProgramList {prefName} {
  global TeXmodeVars menu::items
  
  set "menu::items(TeX program)" $TeXmodeVars(availableTeXPrograms)
  buildLaTeXMenuQuietly
}

proc TeX::updateFormatList {prefName} {
  global TeXmodeVars menu::items
  
  set "menu::items(TeX Format)" $TeXmodeVars(availableTeXFormats)
  buildLaTeXMenuQuietly
}
